package main

import (
	"github.com/veandco/go-sdl2/gfx"
	"github.com/veandco/go-sdl2/sdl"
	"github.com/veandco/go-sdl2/ttf"
)

type Sprite struct {
	surface *sdl.Surface
	texture *sdl.Texture
	width   int32
	height  int32
}

func NewSprite(renderer *sdl.Renderer, surface *sdl.Surface) *Sprite {
	var err error
	sprite := new(Sprite)
	sprite.surface = surface
	sprite.width = sprite.surface.W
	sprite.height = sprite.surface.H
	sprite.texture, err = renderer.CreateTextureFromSurface(sprite.surface)
	if err != nil {
		panic(err)
	}
	return sprite
}

func NewSpriteFromText(renderer *sdl.Renderer, font *ttf.Font, text string, color sdl.Color) *Sprite {
	sprite := new(Sprite)
	var err error
	sprite.surface, err = font.RenderUTF8Blended(text, color)
	if err != nil {
		panic(err)
	}
	sprite.width = sprite.surface.W
	sprite.height = sprite.surface.H
	sprite.texture, err = renderer.CreateTextureFromSurface(sprite.surface)
	if err != nil {
		panic(err)
	}
	return sprite
}

func (s *Sprite) Draw(renderer *sdl.Renderer, x int32, y int32) {
	renderer.Copy(s.texture, nil, &sdl.Rect{
		X: x,
		Y: y,
		W: s.width,
		H: s.height,
	})
}

func (s *Sprite) DrawCenter(renderer *sdl.Renderer, x int32, y int32) {
	s.Draw(renderer, x-(s.Width()/2), y-(s.Height()/2))
}

func (s *Sprite) RotateAndDraw(renderer *sdl.Renderer, angle float64, x int32, y int32) {
	surface := gfx.RotoZoomSurface(s.surface, angle, 1.0, gfx.SMOOTHING_ON)
	s.width = surface.W
	s.height = surface.H
	var err error
	s.texture.Destroy()
	s.texture, err = renderer.CreateTextureFromSurface(surface)
	surface.Free()
	if err != nil {
		panic(err)
	}
	s.Draw(renderer, x, y)
}

func (s *Sprite) RotateAndDrawCenter(renderer *sdl.Renderer, angle float64, x int32, y int32) {
	s.RotateAndDraw(renderer, angle, x-(s.Width()/2), y-(s.Height()/2))
}

func (s *Sprite) Width() int32 {
	return s.width
}

func (s *Sprite) Height() int32 {
	return s.height
}
